/*  CLIPS Version 4.20 4/29/88 */
 
   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                   DEFFACTS MODULE                   */
   /*******************************************************/
   
#include <stdio.h>

#include "setup.h"

#if DEFFACTS_CONSTRUCT

#include "constant.h"
#include "scanner.h"
#include "deffacts.h"
#include "memory.h"
   
/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

  static char                   *df_1st_phase();
  static struct fact            *get_df1();
  static struct fact            *get_df2();
  static int                     df_err1();
  static struct element         *df_2nd_phase();
   struct fact            *df_3rd_phase();
   struct dfact           *find_deffact();
   struct dfact           *get_next_deffact();
   char                   *get_deffact_name();
   char                   *get_deffact_ppform();
   
/****************************************/
/* GLOBAL INTERNAL FUNCTION DEFINITIONS */
/****************************************/
   
   int                     parse_deffacts();
   int                     createinitial();
   int                     returnelements();
   int                     init_deffacts();
   int                     clear_deffacts();
   int                     undeffacts_command();
   int                     list_deffacts();
   int                     list_dfct_command();
   int                     ppdef_command();
   int                     save_deffacts();
   
/****************************************/
/* GLOBAL EXTERNAL FUNCTION DEFINITIONS */
/****************************************/

   extern char            *copy_pp_buffer();
   extern struct fact     *get_el();
   extern struct draw     *add_symbol();
   extern char            *check_name();
   
/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/
  
   static struct token     inp_tkn;
   static int              deffacts_error;
   static struct dfact    *defptr = NULL;  
   static struct dfact    *deflist;

#if RUN_TIME

/*****************************/
/* SET_UP_DEFFACTS:          */
/*****************************/
set_up_deffacts()
  {
   add_reset_function("deffacts",init_deffacts);
  }

#else

/****************************************/
/* GLOBAL EXTERNAL VARIABLE DEFINITIONS */
/****************************************/

   extern int              CONSERVE_SPACE;
   extern int              LOAD_FLAG;

/*****************************/
/* SET_UP_DEFFACTS:          */
/*****************************/
set_up_deffacts()
  {
   createinitial();
   add_reset_function("deffacts",init_deffacts);
   add_clear_function("deffacts",clear_deffacts);
   add_save_function("deffacts",save_deffacts);
   add_construct("deffacts",parse_deffacts);
   define_function("undeffacts",   'v', undeffacts_command, "undeffacts_command");
   define_function("list-deffacts",'v', list_dfct_command,  "list_dfct_command");
   define_function("ppdeffact",    'v', ppdef_command,      "ppdef_command");  
  }

/*****************************/
/* CLEAR_DEFFACTS:          */
/*****************************/
clear_deffacts()
  {
   remove_all_deffacts();
   createinitial();
  }
  
/*****************************/
/* SAVE_DEFFACTS:          */
/*****************************/
save_deffacts(log_name)
  char *log_name;
  {
   struct dfact *def_ptr;
   char *ppform;
   
   def_ptr = get_next_deffact(NULL);
   while (def_ptr != NULL)
     {
      ppform = get_deffact_ppform(def_ptr);
      if (ppform != NULL)
        {
         print_in_chunks(log_name,ppform);
         cl_print(log_name,"\n");
        }
      def_ptr = get_next_deffact(def_ptr);
     }
  }

/***************************************************************/
/* PARSE_DEFFACTS:  The purpose of this function is to parse   */
/*   the deffacts statement into a list of facts which can be  */
/*   asserted when a reset is performed.  The name of the      */ 
/*   deffacts block is saved along with each fact for use with */
/*   the undeffacts statement.                                 */
/***************************************************************/
int parse_deffacts(read_source)
  char *read_source;
  {
   char *df_name;
   struct fact *temp;
   struct dfact *new_dfact;
   
   deffacts_error = FALSE;
   set_pp_buffer_status(ON);

   flush_pp_buffer();            
   save_pp_buffer("(deffacts ");

   /*============================================================*/
   /* Parse the name and comment fields of the rule.  Excise the */
   /* rule if it already exists.                                 */
   /*============================================================*/  

   df_name = df_1st_phase(read_source);
   if (deffacts_error == TRUE) { return(deffacts_error); } 
   
   /*=================================*/
   /* Check that next token is a '('. */
   /*=================================*/

   temp = get_df1(df_name,read_source);

   if (deffacts_error == TRUE) { return(deffacts_error); }

   temp = get_df2(temp);

   new_dfact = get_struct(dfact);
   new_dfact->name = gm2(sizeof(char) * (strlen(df_name) + 1));
   strcpy(new_dfact->name,df_name);
   new_dfact->flist = temp;
   new_dfact->next = NULL;

   if (CONSERVE_SPACE == TRUE)
     { new_dfact->pp_form = NULL; }
   else
     { new_dfact->pp_form = copy_pp_buffer(); } 
   
   if (defptr == NULL)
     { deflist = new_dfact; }
   else
     { defptr->next = new_dfact; }
   defptr = new_dfact;

   return(deffacts_error);       
  }

/*********************************************************************/
/* df_1st_phase:                                                     */
/*********************************************************************/
char *df_1st_phase(read_source)
  char *read_source;
  {
   char *df_name;

   /*===========================================================*/
   /* Get next token, which should be the name of the deffacts. */
   /*===========================================================*/


   gettoken(read_source,&inp_tkn);
   if (inp_tkn.token != WORD)
     { 
      cl_print("werror","\nMissing deffacts name\n");
      deffacts_error = TRUE;
      return(NULL);
     }
   
   df_name = inp_tkn.tknword;

   if ((delete_deffacts(df_name) == TRUE) && (get_rules_watch() == ON))
     {
      cl_print("wdialog","Removing deffacts block ");
      cl_print("wdialog",df_name);
      cl_print("wdialog","\n");
     }

   /*==========================================================*/
   /* If watch rules is on, indicate deffacts being processed. */
   /*==========================================================*/

   if ((get_rules_watch() == ON) && (LOAD_FLAG == TRUE))
     {
      cl_print("wdialog","Processing deffacts block ");
      cl_print("wdialog",df_name);
      cl_print("wdialog","\n");
     }
   else if (LOAD_FLAG == TRUE)
     { cl_print("wdialog","$"); }

   /*===========================*/
   /* Get comment if it exists. */
   /*===========================*/

   save_pp_buffer(" ");                 

   gettoken(read_source,&inp_tkn);
   if (inp_tkn.token == STRING)
     {
      save_pp_buffer("\n   ");           
      gettoken(read_source,&inp_tkn);
     }
   else                                 
     {
      pp_backup();
      save_pp_buffer("\n   ");
      save_pp_buffer(inp_tkn.print_rep);
     }

   return(df_name);
  }

/*********************************************************************/
/* GET_DF1:                                                          */
/*********************************************************************/
 struct fact *get_df1(df_name,read_source)
  char *df_name;
  char *read_source;
  {
   struct fact *list_of_facts;
   struct fact *new_fact, *last_fact;
   struct element *fact_def;

   list_of_facts = NULL;
   last_fact = NULL;

   while (inp_tkn.token == LPAREN)
     {
      fact_def = df_2nd_phase(df_name,read_source);
      if (deffacts_error == TRUE)
        { 
         df_err1(list_of_facts);
         return(NULL);
        }
      new_fact = get_struct(fact);
      new_fact->next = NULL;
      new_fact->previous = NULL;
      new_fact->list = NULL;
      new_fact->atoms = fact_def; 

      if (last_fact == NULL)
        { list_of_facts = new_fact; }
      else
        { last_fact->next = new_fact; }
      last_fact = new_fact;

      save_pp_buffer("\n   ");
      gettoken(read_source,&inp_tkn);
     }

   if (inp_tkn.token == RPAREN)
     {
      pp_backup();
      pp_backup();
      save_pp_buffer(")\n");
      return(list_of_facts);
     }
   
   deffacts_error = TRUE;

   cl_print("werror","\nExpected a ')' to close the deffacts block\n");
   cl_print("werror","or a '(' to begin a new fact definition in deffacts ");
   cl_print("werror",df_name);
   cl_print("werror","\n");

   df_err1(list_of_facts);

   return(NULL);
  }

/*********************************************************************/
/* DF_ERR1:                                                          */
/*********************************************************************/
 int df_err1(list_of_facts)
  struct fact *list_of_facts;
  {
   struct fact *temp;

   while (list_of_facts != NULL)
     {
      temp = list_of_facts->next; 
      if (list_of_facts->atoms != NULL)
        { returnelements(list_of_facts->atoms); }
      rtn_struct(fact,list_of_facts);
      list_of_facts = temp;
     }
  }
 
/*********************************************************************/
/* GET_DF2:                                                          */
/*********************************************************************/
 struct fact *get_df2(fact_ptr)
  struct fact *fact_ptr;
  {
   struct fact *temp, *new_fact, *list_of_facts, *last_fact;

   list_of_facts = NULL;
   last_fact = NULL;

   while (fact_ptr != NULL)
     {
      new_fact = df_3rd_phase(fact_ptr->atoms);

      fact_install(new_fact);

      if (last_fact == NULL)
        { list_of_facts = new_fact; }
      else
        { last_fact->next = new_fact; }
      last_fact = new_fact;

      temp = fact_ptr->next;
      returnelements(fact_ptr->atoms);
      rtn_struct(fact,fact_ptr);
      fact_ptr = temp;
     }

   return(list_of_facts);
  }

/*********************************************************************/
/* df_2nd_phase:                                                     */
/*********************************************************************/
 struct element *df_2nd_phase(df_name,read_source)
  char *df_name;
  char *read_source;
  {
   struct element *first_element, *last_element, *next_element;
  
   first_element = NULL;
   last_element = NULL;
   gettoken(read_source,&inp_tkn);

   while ((inp_tkn.token == WORD) || (inp_tkn.token == NUMBER) || (inp_tkn.token == STRING))
     { 
      next_element = get_struct(element);
      next_element->next = NULL;
      next_element->type = inp_tkn.token;
      if (next_element->type == NUMBER)
        { next_element->val.fvalue = inp_tkn.tknnumber; }
      else
        { next_element->val.hvalue = inp_tkn.hashword; } 
          
      if (last_element == NULL)
        { first_element = next_element; }
      else
        { last_element->next = next_element; }
      last_element = next_element;
      
      save_pp_buffer(" ");
      gettoken(read_source,&inp_tkn);           
     }

   /*=============================================*/
   /* If the fact was not closed with a ')', then */
   /* an error has occured.                       */
   /*=============================================*/

   if (inp_tkn.token == RPAREN)
     {
      if (first_element != NULL)
        {
         pp_backup();
         pp_backup();
         save_pp_buffer(")");
        }
     }
   else 
     {  
      cl_print("werror","\nExpected a ')' to close the fact definition in deffacts ");
      cl_print("werror",df_name);
      cl_print("werror","\n");
      deffacts_error = TRUE;
      if (first_element != NULL) returnelements(first_element);
      return(NULL);
     }

   /*=======================================================*/
   /* If the fact has no fields, then an error has occured. */
   /* Otherwise, attach the linked list of fields to the    */
   /* fact structure.                                       */
   /*=======================================================*/

   if (first_element == NULL)
     {
      cl_print("werror","\nNull fact in deffacts block ");
      cl_print("werror",df_name);
      cl_print("werror","\n");
      deffacts_error = TRUE;
      return(NULL);
     }

   return(first_element);
  }

/*********************************************************************/
/* df_3rd_phase:                                                     */
/*********************************************************************/
struct fact *df_3rd_phase(first_element)
  struct element *first_element;
  {
   struct element *elem_ptr, *new_ptr;
   struct fact *temp;
   int count;

   elem_ptr = first_element;
   count = 0;

   while (elem_ptr != NULL)
     {
      count++;
      elem_ptr = elem_ptr->next;
     }

   temp = get_el(count);
   temp->list = NULL;
   temp->previous = NULL;
   temp->next = NULL;

   elem_ptr = first_element;
   new_ptr = temp->atoms;
   count = 0;

   while (elem_ptr != NULL)
     {
      new_ptr[count].val = elem_ptr->val;
      new_ptr[count].type = elem_ptr->type;
         
      count++;
      elem_ptr = elem_ptr->next;
     }

   return(temp);
  }
  
#endif 

/**************************************************************/
/* init_deffacts:  Copies the deffacts list to the fact list. */
/**************************************************************/
init_deffacts()
  {
   struct dfact *def_ptr;
   struct fact *tempfact, *fact_ptr;
   struct element *elema_ptr, *elemb_ptr;
   int i;

   /*======================================*/
   /* Copy facts from deflist to factlist. */
   /*======================================*/

   def_ptr = deflist;
   while (def_ptr != NULL)
     {
      fact_ptr = def_ptr->flist;

      while (fact_ptr != NULL)
        {
         /*===================================*/
         /* Make a copy of the deffacts fact. */
         /*===================================*/

         tempfact = get_el(fact_ptr->fact_length);
         tempfact->list = NULL;
         tempfact->next = NULL;

         /*===========================*/
         /* Copy individual elements. */
         /*===========================*/

         elema_ptr = fact_ptr->atoms;
         elemb_ptr = tempfact->atoms;
         for (i = 0 ; i < fact_ptr->fact_length ; i++)
           {
            elemb_ptr[i].val = elema_ptr[i].val;
            elemb_ptr[i].type = elema_ptr[i].type;
           }

         /*============================*/
         /* Place fact into fact list. */
         /*============================*/

         add_fact(tempfact);
         fact_ptr = fact_ptr->next;
        }
      def_ptr = def_ptr->next;
     }
  }
  
#if ! RUN_TIME

/*************************************************************/
/* createinitial:  Creates the initial fact, (initial-fact), */
/*   and places it on the deffacts list.                     */
/*************************************************************/
createinitial()
  {
   struct fact *new_fact;
   struct element *elem_ptr;
   struct dfact *new_dfact;

   /*==============================*/
   /* Set fact ID counter to zero. */
   /*==============================*/

   set_fact_id( (FACT_ID) 0);                           

   /*============================================================*/
   /* Create the initial fact and place it on the deffacts list. */
   /*============================================================*/
  
   new_fact = get_el(1);
   new_fact->previous = NULL;
   new_fact->next = NULL;
   new_fact->ID = 0;

   elem_ptr = new_fact->atoms;

   elem_ptr[0].val.hvalue = add_symbol("initial-fact");
   elem_ptr[0].type = WORD;
   elem_ptr[0].next = NULL;

   /*================================================*/
   /* Set the variable which points to the last fact */
   /* added to the deffacts list.                    */
   /*================================================*/

   fact_install(new_fact);

   new_dfact = get_struct(dfact);
   new_dfact->name = gm2(sizeof(char) * (strlen("initial-fact") + 1));
   strcpy(new_dfact->name,"initial-fact");
   new_dfact->flist = new_fact;
   new_dfact->next = NULL;
   new_dfact->pp_form = NULL;

   defptr = deflist = new_dfact;  
  }

/***************************************************************/
/* REMOVE_ALL_DEFFACTS:                  */
/***************************************************************/
remove_all_deffacts()
  {
   struct fact *fact_ptr, *next_fact;
   struct dfact *next_def;
   
   while (deflist != NULL)
     {
      fact_ptr = deflist->flist;
      while (fact_ptr != NULL)
        {
         next_fact = fact_ptr->next;
         fact_deinstall(fact_ptr);
         rtn_el(fact_ptr);
         fact_ptr = next_fact;
        }

      next_def = deflist->next;

      rm(deflist->name,sizeof(char) * (strlen(deflist->name) + 1));
      if (deflist->pp_form != NULL)
        {
         rm(deflist->pp_form,
                    sizeof(char) * (strlen(deflist->pp_form) + 1));
        }
      rtn_struct(dfact,deflist);

      deflist = next_def;
     }
   defptr = NULL;
  }
  
/*****************************************************/
/* delete_deffacts: Delete all facts in the deffacts */
/*   list which have the identifying name of the     */
/*   deffacts block to be removed.                   */
/*****************************************************/
delete_deffacts(deffacts_name)
   char *deffacts_name;
  {
   struct dfact *next_deffact, *last_deffact;
   struct dfact *cur_deffact;
   struct fact *fact_ptr, *next_fact;

   last_deffact = NULL;
   cur_deffact = deflist;
   while (cur_deffact != NULL)
     {
      next_deffact = cur_deffact->next;
      if (strcmp(cur_deffact->name,deffacts_name) == 0)
        {
         fact_ptr = cur_deffact->flist;
         while (fact_ptr != NULL)
           {
            next_fact = fact_ptr->next;
            fact_deinstall(fact_ptr);
            rtn_el(fact_ptr);
            fact_ptr = next_fact;
           }

         if (last_deffact == NULL)
           { deflist = next_deffact; }
         else
           { last_deffact->next = next_deffact; }

         if (defptr == cur_deffact)
           { defptr = last_deffact; }

         rm(cur_deffact->name,sizeof(char) * (strlen(cur_deffact->name) + 1));
         if (cur_deffact->pp_form != NULL)
           {
            rm(cur_deffact->pp_form,
                    sizeof(char) * (strlen(cur_deffact->pp_form) + 1));
           }
         rtn_struct(dfact,cur_deffact);
         return(1);
        }
  
      last_deffact = cur_deffact;
      cur_deffact = next_deffact;
     }

   return(0); 
  }

#endif

/******************************************************************/
/* FIND_DEFFACT:  Searches for a deffact in the list of deffacts. */
/*   Returns a pointer to the deffact if found, otherwise NULL.   */
/******************************************************************/
struct dfact *find_deffact(df_name)
  char *df_name;
  {
   struct dfact *df_ptr;

   df_ptr = deflist;
   while (df_ptr != NULL)
     {
      if (strcmp(df_ptr->name,df_name) == 0)
        { return(df_ptr); }
      
      df_ptr = df_ptr->next; 
     }

   return(NULL);
  }

/************************************************************/
/* SET_DEFLIST:                            */
/************************************************************/
set_deflist(def_ptr)
  struct dfact *def_ptr;
  {
   deflist = def_ptr; 
  }
  
/************************************************************/
/* get_next_deffact:                            */
/************************************************************/
struct dfact *get_next_deffact(def_ptr)
  struct dfact *def_ptr;
  {
   if (def_ptr == NULL)
     { return(deflist); }
   else
     { return(def_ptr->next); }
  }

/************************************************************/
/* get_deffact_name:                            */
/************************************************************/
char *get_deffact_name(def_ptr)
  struct dfact *def_ptr;
  { return(def_ptr->name); }
  
/************************************************************/
/* get_deffact_ppform:                            */
/************************************************************/
char *get_deffact_ppform(def_ptr)
  struct dfact *def_ptr;
  { return(def_ptr->pp_form); }

/**********************************************************/
/* pp_deffact: the driver which actually does the pretty  */
/*   printing of the deffact.                             */
/**********************************************************/
pp_deffact(df_name,fileid)
  char *df_name, *fileid;
  {
   struct dfact *df_ptr;

   df_ptr = find_deffact(df_name);
   if (df_ptr == NULL)        
     {
      cl_print("werror","Unable to find deffact ");
      cl_print("werror",df_name);
      cl_print("werror","\n");
      return(FALSE);
     }
     
   if (get_deffact_ppform(df_ptr) == NULL) return(TRUE);
   print_in_chunks(fileid,get_deffact_ppform(df_ptr));
   return(TRUE);
  }
  
#if ! RUN_TIME

/*****************************************************/
/* undeffacts_command: removes a deffacts statement. */
/*   Syntax: (undeffacts <deffacts name>)            */
/*****************************************************/
int undeffacts_command()
  {
   char *deffacts_name;

   deffacts_name = check_name(1,1,"undeffacts","deffacts name");
   if (deffacts_name == NULL) return(0);

   if (delete_deffacts(deffacts_name) == FALSE)
     { 
      cl_print("werror","Unable to find deffacts block named ");
      cl_print("werror",deffacts_name);
      cl_print("werror","\n");
      return(0);
     }

   return(1);
  }

/************************************************************/
/* ppdef_command: pretty prints a deffact.                  */
/*   Syntax: (pp_deffact <deffact name>)                    */
/************************************************************/
int ppdef_command()
  {
   char *df_name;

   df_name = check_name(1,1,"ppdeffact","deffact name");
   if (df_name == NULL) return(0);

   pp_deffact(df_name,"wdisplay");

   return(1);
  }

/************************************************************/
/* list_deffacts: displays the list of deffacts.            */
/*   Syntax: (list_deffacts)                                */
/************************************************************/
int list_deffacts()
  {
   struct dfact *df_ptr;

   df_ptr = get_next_deffact(NULL);
   while (df_ptr != NULL)
     {
      cl_print("wdisplay",get_deffact_name(df_ptr));
      cl_print("wdisplay","\n");
      df_ptr = get_next_deffact(df_ptr);
     }

   return(1);
  }

/************************************************************/
/* list_dfct_command: displays the list of deffacts.        */
/*   Syntax: (list_deffacts)                                */
/************************************************************/
list_dfct_command()
  {
   if (arg_num_check("list-deffacts",EXACTLY,0) == -1) return(0);

   list_deffacts();
  }

  
#else

   int                        undeffacts_command() {};
   int                        list_dfct_command() {};
   int                        ppdef_command() {};
   
#endif

  
#endif
  
